<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

use App\Models\Product;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\ProductSizeVariation;
use App\Models\Setting;
use App\Models\SmsTemplate;
use App\Models\Api;
use App\Models\Expense;
use App\Models\PaymentRequest;
use App\Models\SteadFast;
use App\Models\Redx;
use App\Models\Pathao;
use App\Models\Courier;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use Auth;
use Illuminate\Support\Facades\Hash;
use Carbon;
use PDF;
use DB;
use Illuminate\Support\Str;

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $info = SteadFast::first();
        // Start with all orders
        $query = Order::orderBy("created_at", "DESC");

        // Check if the current user is a staff member
        if ($request->user()->role == "staff") {
            // If yes, filter orders assigned to this staff member
            $query->where("assign_user_id", $request->user()->id);
        }

        // Paginate the results
        $products = $query->paginate(10);

        // Pass data to the view
        return view("backend.sales.index", compact("products", "info"));
    }

    public function getStatusByCid($id)
    {
        // Fetch API keys and API URL from database
        $info = SteadFast::first();

        if (!$info) {
            return response()->json([
                'status' => 'error',
                'message' => 'API keys not found.'
            ], 404);
        }

        // Fetch delivery status from external API
        $status = $this->fetchStatusFromExternalAPI($info, $id);

        if ($status !== null) {
            return response()->json([
                'status' => 'success',
                'delivery_status' => $status
            ]);
        } else {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fetch delivery status.'
            ], 404);
        }
    }

    /**
     * Fetch delivery status from external API.
     *
     * @param  \App\Models\SteadFast  $info  SteadFast model instance containing API credentials
     * @param  int  $id  Consignment ID (cid)
     * @return string|null
     */
    private function fetchStatusFromExternalAPI($info, $id)
    {
        try {
            $response = Http::withHeaders([
                'Api-Key' => $info->steadfast_api_key,
                'Secret-Key' => $info->steadfast_secret_key,
            ])->get($info->api_url . 'status_by_cid/' . $id);

            if ($response->successful()) {
                $data = $response->json();
                return $data['delivery_status'] ?? null;
            } else {
                return null;
            }
        } catch (\Exception $e) {
            // Handle exceptions if any
            \Log::error('Exception occurred while fetching delivery status: ' . $e->getMessage());
            return null;
        }
    }



    public function facebook()
    {
        $products = Order::where("fbwaorder", "facebook_order")
            ->orderBy("created_at", "DESC")
            ->paginate(10);

        return view("backend.sales.facebook")->with("products", $products);
    }

    public function status_wise_order(Request $request)
    {
        try {
            // Retrieve the status value from the request
            $statusValue = $request->input("status");

            // Perform a query to fetch orders based on the provided status value
            $received_status_order = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->where("orders.status", $statusValue)
                ->select("order_details.*", "orders.*")
                ->get();

            //$received_status_order = DB::table('Order')->join('orders','order_details.order_id','=','orders.id')->where('orders.status', '$statusValue')->get();

            // Start with all orders
            $query = Order::where("status", $statusValue)->orderBy(
                "created_at",
                "DESC"
            );

            // Check if the current user is a staff member
            if ($request->user()->role == "staff") {
                // If yes, filter orders assigned to this staff member
                $query->where("assign_user_id", $request->user()->id);
            }

            // Paginate the results
            $products = $query->paginate(25);

            // Render the view
            $view = view(
                "backend.sales.status_received_order",
                compact("received_status_order", "products")
            )->render();

            // Return the JSON response with the rendered view
            return response()->json(["success" => true, "view" => $view]);
        } catch (\Exception $e) {
            // If an exception occurs, return an error response
            return response()->json([
                "success" => false,
                "error" => $e->getMessage(),
            ]);
        }
    }


    public function report(Request $request)
    {
        $data = [];
        $created_at = Carbon\Carbon::today();
        $created_at = $created_at->format("d-m-Y");
        $fromDate = $created_at;
        $todate = $created_at;
        $statusNew = $request->status;

        $dataasd = DB::table("order_details")
            ->join("orders", "order_details.order_id", "=", "orders.id")
            ->whereIn("orders.status", [
                "Confirm",
                "Shipped to Courier",
                "New order",
                "Delivery Done",
            ])
            ->get();

        foreach ($dataasd as $key => $value) {
            $prcasd = Product::where("id", $value->product_id)
                ->withTrashed()
                ->first();
            $data[$key]["id"] = $value->product_id;
            $data[$key]["product_name"] = $prcasd->title ?? 'Product Deleted';
            $data[$key]["sku"] = $prcasd->SKU ?? '';
            $data[$key]["size"] = $value->size;
            $data[$key]["color"] = $value->color;
            $data[$key]["quantity"] = $value->quantity;
        }

        $uniqueProducts = [];

        // Loop through each product in the data array and combine the quantities
        foreach ($data as $product) {
            $key =
                $product["id"] .
                "-" .
                $product["size"] .
                "-" .
                $product["color"];

            if (isset($uniqueProducts[$key])) {
                $uniqueProducts[$key]["quantity"] += $product["quantity"];
            } else {
                $uniqueProducts[$key] = $product;
            }
        }

        // Sort the array by product name
        usort($uniqueProducts, function ($a, $b) {
            return $a["product_name"] <=> $b["product_name"];
        });
        $data = $uniqueProducts;
        return view(
            "backend.report.index",
            compact("data", "fromDate", "todate", "statusNew")
        );
    }
    public function reportdate(Request $request)
    {
        $data = [];
        $fromDate = $request->fromDate;
        $todate = $request->todate;
        $statusNew = $request->status;

        $dataasd = DB::table("order_details")
            ->join("orders", "order_details.order_id", "=", "orders.id")
            ->where("orders.status", $statusNew)
            ->whereBetween("orders.created_at", [
                Carbon\Carbon::parse($fromDate)->startOfDay(),
                Carbon\Carbon::parse($todate)->endOfDay(),
            ])
            ->get();

        foreach ($dataasd as $key => $value) {
            $prcasd = Product::where("id", $value->product_id)
                ->withTrashed()
                ->first();
            $data[$key]["id"] = $value->product_id;
            $data[$key]["product_name"] = $prcasd->title ?? 'Product Deleted';
            $data[$key]["sku"] = $prcasd->SKU ?? '';
            $data[$key]["size"] = $value->size;
            $data[$key]["color"] = $value->color;
            $data[$key]["quantity"] = $value->quantity;
        }
        $uniqueProducts = [];

        // Loop through each product in the data array and combine the quantities
        foreach ($data as $product) {
            $key =
                $product["id"] .
                "-" .
                $product["size"] .
                "-" .
                $product["color"];

            if (isset($uniqueProducts[$key])) {
                $uniqueProducts[$key]["quantity"] += $product["quantity"];
            } else {
                $uniqueProducts[$key] = $product;
            }
        }

        // Sort the array by product name
        usort($uniqueProducts, function ($a, $b) {
            return $a["product_name"] <=> $b["product_name"];
        });
        $data = $uniqueProducts;
        return view(
            "backend.report.index",
            compact("data", "fromDate", "todate", "statusNew")
        );
    }
    public function search($status, Request $request)
    {
        // Start with all orders
        $query = Order::orderBy("created_at", "DESC")->where("status", $status);

        // Check if the current user is a staff member
        if ($request->user()->role == "staff") {
            // If yes, filter orders assigned to this staff member
            $query->where("assign_user_id", $request->user()->id);
        }

        // Paginate the results
        $products = $query->paginate(20);

        return view("backend.sales.index", compact("products"));
    }
    public function downloadExcel()
    {
        $date = \Carbon\Carbon::today();
        $date = $date->format("Y-m-d h:i A");
        $products = Order::all();
        $pdf = PDF::loadView(
            "backend.sales.report",
            compact("products", "date")
        );

        return $pdf->download("orders.pdf");
    }
    public function downloadorder(Request $request)
    {
        if (isset($request->date)) {
            $date = $request->date;

            $products = Order::where("status", $request->status)
                ->whereDay("created_at", $date)
                ->get();
        } else {
            $date = \Carbon\Carbon::today();
            $date = $date->format("Y-m-d h:i A");
            $products = Order::where("status", $request->status)->get();
        }

        $pdf = PDF::loadView(
            "backend.sales.report",
            compact("products", "date")
        );

        return $pdf->download("orders.pdf");
    }
    public function bulkStatusChange(Request $request)
    {
        // Retrieve the selected status and order IDs from the form submission
        $newStatus = $request->input("new_status");
        $orderIds = $request->input("order_ids");


        //dd($newStatus);
        if (gettype($orderIds) != "array") {
            request()
                ->session()
                ->flash("error", "Select Order First");
            return back();
        }
        // Update the status of selected orders
        Order::whereIn("id", $orderIds)->update(["status" => $newStatus]);

        // Optionally, redirect back or return a response
        return redirect()
            ->back()
            ->with("success", "Status updated successfully");
    }
    public function getsearch(Request $request)
    {
        $search = $request->search;

        if (empty($search)) {
            // Redirect to a specific route or page if search term is empty
            return redirect()->route("order.index");
        }
        $set = Setting::find(1); // Assuming the setting ID is 1

        $products = Order::where("id", "like", "%" . $request->search . "%")
            ->orWhere("first_name", "like", "%" . $request->search . "%")
            ->orWhere("phone", "like", "%" . $request->search . "%")
            ->orWhere("address", "like", "%" . $request->search . "%")
            ->orWhere("note", "like", "%" . $request->search . "%")
            ->orWhere("ip_address", "like", "%" . $request->search . "%")
            ->orderBy("id", "DESC")
            ->paginate(20);

        return view("backend.sales.index", compact("products"));
    }

    public function fullreportdate(Request $request)
    {
        $action = $request->action;
        if ($action == "search") {
            $created_at = Carbon\Carbon::today();
            $created_at = $created_at->format("d-m-Y");
            $fromDate = $request->fromDate;
            $todate = $request->todate;

            $total_order = Order::whereBetween("orders.created_at", [
                Carbon\Carbon::parse($fromDate)->startOfDay(),
                Carbon\Carbon::parse($todate)->endOfDay(),
            ])->count();

            $total_deliver = Order::where("status", "Delivery Done")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->count();
            $total_pending = Order::whereNotIn("status", [
                "Cancel",
                "Delivery Done",
            ])
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->count();
            $total_cancel = Order::where("status", "Cancel")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->count();
            $productinsale = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("orders.status")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->sum("order_details.quantity");
            $productsale = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("orders.status")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->whereNotIn("orders.status", ["Cancel", "Delivery Done"])
                ->sum("order_details.quantity");
            $productdelivery = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("orders.status")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->where("orders.status", "Delivery Done")
                ->sum("order_details.quantity");
            $productCancel = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("orders.status")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->where("orders.status", "Cancel")
                ->sum("order_details.quantity");

            $costprice = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->where("orders.status", "Delivery Done")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->sum(DB::raw("order_details.cost * order_details.quantity"));
            $totalprice = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("orders.status")
                ->where("orders.status", "Delivery Done")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->sum(DB::raw("orders.sub_total"));

            $grossprofit = $totalprice - $costprice;

            $totaldiscount = DB::table("orders")
                ->where("status", "Delivery Done")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->sum("discount");
            $totalloss = Order::where("status", "Cancel")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->sum("shipping_cost");
            $totalexpanse = Expense::whereBetween("expenses.created_at", [
                Carbon\Carbon::parse($fromDate)->startOfDay(),
                Carbon\Carbon::parse($todate)->endOfDay(),
            ])->sum("amount");
            $grosssale = $totalprice - $costprice - $totaldiscount - $totalloss;
            $netprofit =
                $totalprice -
                $costprice -
                $totaldiscount -
                $totalloss -
                $totalexpanse;

            $data = [];

            $dataasd = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->where("orders.status", "Delivery Done")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->get();

            foreach ($dataasd as $key => $value) {
                $prcasd = Product::where("id", $value->product_id)->first();
                $data[$key]["id"] = $value->product_id;
                $data[$key]["product_name"] = $prcasd->title ?? 'Product Deleted';
                $data[$key]["sku"] = $prcasd->SKU ?? '';
                $data[$key]["size"] = $value->size ?? '';
                $data[$key]["color"] = $value->color ?? '';
                $data[$key]["quantity"] = $value->quantity ?? '';
            }
            $uniqueProducts = [];

            // Loop through each product in the data array and combine the quantities
            foreach ($data as $product) {
                $key =
                    $product["id"] .
                    "-" .
                    $product["size"] .
                    "-" .
                    $product["color"];

                if (isset($uniqueProducts[$key])) {
                    $uniqueProducts[$key]["quantity"] += $product["quantity"];
                } else {
                    $uniqueProducts[$key] = $product;
                }
            }

            // Sort the array by product name
            usort($uniqueProducts, function ($a, $b) {
                return $a["product_name"] <=> $b["product_name"];
            });
            $data = $uniqueProducts;

            return view(
                "backend.report.report",
                compact(
                    "fromDate",
                    "data",
                    "productCancel",
                    "productinsale",
                    "productdelivery",
                    "total_pending",
                    "todate",
                    "total_order",
                    "total_deliver",
                    "total_cancel",
                    "productsale",
                    "costprice",
                    "totalprice",
                    "totaldiscount",
                    "totalloss",
                    "totalexpanse",
                    "grosssale",
                    "grossprofit",
                    "netprofit"
                )
            );
        } elseif ($action == "download") {
            $created_at = Carbon\Carbon::today();
            $created_at = $created_at->format("d-m-Y");
            $fromDate = $request->fromDate;
            $todate = $request->todate;

            $total_order = Order::whereBetween("orders.created_at", [
                Carbon\Carbon::parse($fromDate)->startOfDay(),
                Carbon\Carbon::parse($todate)->endOfDay(),
            ])->count();

            $total_deliver = Order::where("status", "Delivery Done")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->count();
            $total_pending = Order::whereNotIn("status", [
                "Cancel",
                "Delivery Done",
            ])
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->count();
            $total_cancel = Order::where("status", "Cancel")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->count();
            $productinsale = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("orders.status")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->sum("order_details.quantity");
            $productsale = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("orders.status")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->whereNotIn("orders.status", ["Cancel", "Delivery Done"])
                ->sum("order_details.quantity");
            $productdelivery = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("orders.status")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->where("orders.status", "Delivery Done")
                ->sum("order_details.quantity");
            $productCancel = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("orders.status")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->where("orders.status", "Cancel")
                ->sum("order_details.quantity");

            $costprice = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->where("orders.status", "Delivery Done")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->sum(DB::raw("order_details.cost * order_details.quantity"));
            $totalprice = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("orders.status")
                ->where("orders.status", "Delivery Done")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->sum(DB::raw("order_details.price * order_details.quantity"));

            //$costpriceas = OrderDetail::get();

            //$totalprice = Order::where('status','Delivery Done')->sum('sub_total');
            $grossprofit = $totalprice - $costprice;
            $totaldiscount = DB::table("orders")
                ->where("status", "Delivery Done")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->sum("discount");
            //$totaldiscount = Order::sum('discount');
            $totalloss = Order::where("status", "Cancel")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->sum("shipping_cost");
            $totalexpanse = Expense::whereBetween("expenses.created_at", [
                Carbon\Carbon::parse($fromDate)->startOfDay(),
                Carbon\Carbon::parse($todate)->endOfDay(),
            ])->sum("amount");
            $grosssale = $totalprice - $costprice - $totaldiscount - $totalloss;
            $netprofit =
                $totalprice -
                $costprice -
                $totaldiscount -
                $totalloss -
                $totalexpanse;

            $data = [];

            $dataasd = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->where("orders.status", "Delivery Done")
                ->whereBetween("orders.created_at", [
                    Carbon\Carbon::parse($fromDate)->startOfDay(),
                    Carbon\Carbon::parse($todate)->endOfDay(),
                ])
                ->get();

            foreach ($dataasd as $key => $value) {
                $prcasd = Product::where("id", $value->product_id)->first();
                $data[$key]["id"] = $value->product_id;
                $data[$key]["product_name"] = $prcasd->title;
                $data[$key]["sku"] = $prcasd->SKU;
                $data[$key]["size"] = $value->size;
                $data[$key]["color"] = $value->color;
                $data[$key]["quantity"] = $value->quantity;
            }
            $uniqueProducts = [];

            // Loop through each product in the data array and combine the quantities
            foreach ($data as $product) {
                $key =
                    $product["id"] .
                    "-" .
                    $product["size"] .
                    "-" .
                    $product["color"];

                if (isset($uniqueProducts[$key])) {
                    $uniqueProducts[$key]["quantity"] += $product["quantity"];
                } else {
                    $uniqueProducts[$key] = $product;
                }
            }

            // Sort the array by product name
            usort($uniqueProducts, function ($a, $b) {
                return $a["product_name"] <=> $b["product_name"];
            });
            $data = $uniqueProducts;
            $settings = Setting::where("id", 1)->first();
            $pdf = PDF::loadView(
                "backend.report.pdf",
                compact(
                    "fromDate",
                    "settings",
                    "data",
                    "productCancel",
                    "productinsale",
                    "productdelivery",
                    "total_pending",
                    "todate",
                    "total_order",
                    "total_deliver",
                    "total_cancel",
                    "productsale",
                    "costprice",
                    "totalprice",
                    "totaldiscount",
                    "totalloss",
                    "totalexpanse",
                    "grosssale",
                    "grossprofit",
                    "netprofit"
                )
            );
            return $pdf->download(
                "report (" . $fromDate . " to " . $todate . ").pdf"
            );
        } elseif ($action == "fullreport") {
            $created_at = Carbon\Carbon::today();
            $created_at = $created_at->format("d-m-Y");
            $fromDate = $created_at;
            $todate = $created_at;

            $total_order = Order::count();

            $total_deliver = Order::where("status", "Delivery Done")->count();
            $total_pending = Order::whereNotIn("status", [
                "Cancel",
                "Delivery Done",
            ])->count();
            $total_cancel = Order::where("status", "Cancel")->count();
            $productinsale = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("order.status")
                ->sum("order_details.quantity");
            $productsale = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("order.status")
                ->whereNotIn("orders.status", ["Cancel", "Delivery Done"])
                ->sum("order_details.quantity");
            $productdelivery = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("order.status")
                ->where("orders.status", "Delivery Done")
                ->sum("order_details.quantity");
            $productCancel = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("order.status")
                ->where("orders.status", "Cancel")
                ->sum("order_details.quantity");

            $costprice = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->where("orders.status", "Delivery Done")
                ->sum(DB::raw("order_details.cost * order_details.quantity"));
            $totalprice = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->select("order.status")
                ->where("orders.status", "Delivery Done")
                ->sum(DB::raw("order_details.price * order_details.quantity"));

            //$costpriceas = OrderDetail::get();

            //$totalprice = Order::where('status','Delivery Done')->sum('sub_total');
            $grossprofit = $totalprice - $costprice;
            $totaldiscount = DB::table("orders")
                ->where("status", "Delivery Done")
                ->sum("discount");
            //$totaldiscount = Order::sum('discount');
            $totalloss = Order::where("status", "Cancel")->sum("shipping_cost");
            $totalexpanse = Expense::sum("amount");
            $grosssale = $totalprice - $costprice - $totaldiscount - $totalloss;
            $netprofit =
                $totalprice -
                $costprice -
                $totaldiscount -
                $totalloss -
                $totalexpanse;

            $data = [];

            $dataasd = DB::table("order_details")
                ->join("orders", "order_details.order_id", "=", "orders.id")
                ->where("orders.status", "Delivery Done")
                ->get();

            foreach ($dataasd as $key => $value) {
                $prcasd = Product::where("id", $value->product_id)->first();
                $data[$key]["id"] = $value->product_id;
                $data[$key]["product_name"] = $prcasd->title;
                $data[$key]["sku"] = $prcasd->SKU;
                $data[$key]["size"] = $value->size;
                $data[$key]["color"] = $value->color;
                $data[$key]["quantity"] = $value->quantity;
            }
            $uniqueProducts = [];

            // Loop through each product in the data array and combine the quantities
            foreach ($data as $product) {
                $key =
                    $product["id"] .
                    "-" .
                    $product["size"] .
                    "-" .
                    $product["color"];

                if (isset($uniqueProducts[$key])) {
                    $uniqueProducts[$key]["quantity"] += $product["quantity"];
                } else {
                    $uniqueProducts[$key] = $product;
                }
            }

            // Sort the array by product name
            usort($uniqueProducts, function ($a, $b) {
                return $a["product_name"] <=> $b["product_name"];
            });
            $data = $uniqueProducts;
            $settings = Setting::where("id", 1)->first();
            $pdf = PDF::loadView(
                "backend.report.all",
                compact(
                    "fromDate",
                    "settings",
                    "data",
                    "productCancel",
                    "productinsale",
                    "productdelivery",
                    "total_pending",
                    "todate",
                    "total_order",
                    "total_deliver",
                    "total_cancel",
                    "productsale",
                    "costprice",
                    "totalprice",
                    "totaldiscount",
                    "totalloss",
                    "totalexpanse",
                    "grosssale",
                    "grossprofit",
                    "netprofit"
                )
            );
            return $pdf->download("full-report-" . $created_at . ".pdf");
        }
    }
    public function fullreport()
    {
        $created_at = Carbon\Carbon::today();
        $created_at = $created_at->format("d-m-Y");
        $fromDate = $created_at;
        $todate = $created_at;

        $total_order = Order::count();

        $total_deliver = Order::where("status", "Delivery Done")->count();
        $total_pending = Order::whereNotIn("status", [
            "Cancel",
            "Delivery Done",
        ])->count();
        $total_cancel = Order::where("status", "Cancel")->count();
        $productinsale = DB::table("order_details")
            ->join("orders", "order_details.order_id", "=", "orders.id")
            ->select("order.status")
            ->sum("order_details.quantity");
        $productsale = DB::table("order_details")
            ->join("orders", "order_details.order_id", "=", "orders.id")
            ->select("order.status")
            ->whereNotIn("orders.status", ["Cancel", "Delivery Done"])
            ->sum("order_details.quantity");
        $productdelivery = DB::table("order_details")
            ->join("orders", "order_details.order_id", "=", "orders.id")
            ->select("order.status")
            ->where("orders.status", "Delivery Done")
            ->sum("order_details.quantity");
        $productCancel = DB::table("order_details")
            ->join("orders", "order_details.order_id", "=", "orders.id")
            ->select("order.status")
            ->where("orders.status", "Cancel")
            ->sum("order_details.quantity");

        $costprice = DB::table("order_details")
            ->join("orders", "order_details.order_id", "=", "orders.id")
            ->where("orders.status", "Delivery Done")
            ->sum(DB::raw("order_details.cost * order_details.quantity"));
        $totalprice = DB::table("order_details")
            ->join("orders", "order_details.order_id", "=", "orders.id")
            ->select("order.status")
            ->where("orders.status", "Delivery Done")
            ->sum(DB::raw("orders.sub_total"));

        //$costpriceas = OrderDetail::get();

        //$totalprice = Order::where('status','Delivery Done')->sum('sub_total');
        $grossprofit = $totalprice - $costprice;
        $totaldiscount = DB::table("orders")
            ->where("status", "Delivery Done")
            ->sum("discount");
        //$totaldiscount = Order::sum('discount');
        $totalloss = Order::where("status", "Cancel")->sum("shipping_cost");
        $totalexpanse = Expense::sum("amount");
        $grosssale = $totalprice - $costprice - $totaldiscount - $totalloss;
        $netprofit =
            $totalprice -
            $costprice -
            $totaldiscount -
            $totalloss -
            $totalexpanse;

        $data = [];

        $dataasd = DB::table("order_details")
            ->join("orders", "order_details.order_id", "=", "orders.id")
            ->where("orders.status", "Delivery Done")
            ->get();

        foreach ($dataasd as $key => $value) {
            $prcasd = Product::where("id", $value->product_id)->first();
            $data[$key]["id"] = $value->product_id;
            $data[$key]["product_name"] = $prcasd->title;
            $data[$key]["sku"] = $prcasd->SKU;
            $data[$key]["size"] = $value->size;
            $data[$key]["color"] = $value->color;
            $data[$key]["quantity"] = $value->quantity;
        }
        $uniqueProducts = [];

        // Loop through each product in the data array and combine the quantities
        foreach ($data as $product) {
            $key =
                $product["id"] .
                "-" .
                $product["size"] .
                "-" .
                $product["color"];

            if (isset($uniqueProducts[$key])) {
                $uniqueProducts[$key]["quantity"] += $product["quantity"];
            } else {
                $uniqueProducts[$key] = $product;
            }
        }

        // Sort the array by product name
        usort($uniqueProducts, function ($a, $b) {
            return $a["product_name"] <=> $b["product_name"];
        });
        $data = $uniqueProducts;

        return view(
            "backend.report.reportall",
            compact(
                "fromDate",
                "data",
                "productCancel",
                "productinsale",
                "productdelivery",
                "total_pending",
                "todate",
                "total_order",
                "total_deliver",
                "total_cancel",
                "productsale",
                "costprice",
                "totalprice",
                "totaldiscount",
                "totalloss",
                "totalexpanse",
                "grosssale",
                "grossprofit",
                "netprofit"
            )
        );
    }
    public function status($id, $status)
    {
        $order = Order::findOrFail($id);

        $order->status = $status;
        $smsapi = Api::where("id", 1)->first();
        if ($smsapi->sms_status == "on") {
            $settings = Setting::where("id", 1)->first();
            $smstemp = SmsTemplate::where("sms_for", $status)->first();
            if (isset($smstemp)) {
                if ($smstemp->status == "on") {
                    $message = $smstemp->message;
                    $orderprefix = $settings->order_prefix;
                    $orderid = $settings->order_prefix . $order->id;
                    $amount = $order->total_amount;
                    $sub_total = $order->sub_total;
                    $discount = $order->discount;
                    $collectables = $order->payment - $order->total_amount;
                    $shippingcost = $order->shipping_cost;
                    $first_name = $order->first_name;
                    $phone = $order->phone;
                    $address = $order->address;

                    $message = str_replace("{id}", $orderid, $message);
                    $message = str_replace("{amount}", $amount, $message);
                    $message = str_replace("{sub_total}", $sub_total, $message);
                    $message = str_replace("{discount}", $discount, $message);
                    $message = str_replace(
                        "{shipping_cost}",
                        $shippingcost,
                        $message
                    );
                    $message = str_replace("{name}", $first_name, $message);
                    $message = str_replace("{phone}", $phone, $message);
                    $message = str_replace("{address}", $address, $message);
                    $message = str_replace(
                        "{collectables}",
                        $collectables,
                        $message
                    );

                    $phonenumber = $order->phone;
                    $message = urlencode($message);
                    $phoneNumber = "88" . $phonenumber; // The phone number to send the message to
                    //dd($phoneNumber);
                    $senderId = $smsapi->sms_sender_id; // Your approved Sender ID
                    $apiKey = $smsapi->sms_api_key; // Your API key

                    // Build the URL
                    $url =
                        "https://isms.mimsms.com/smsapi?api_key=" .
                        $apiKey .
                        "&type=text&contacts=" .
                        $phoneNumber .
                        "&senderid=" .
                        $senderId .
                        "&msg=" .
                        $message;

                    // Initialize cURL
                    $ch = curl_init();

                    // Set cURL options
                    curl_setopt($ch, CURLOPT_URL, $url);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                    // Execute the request
                    $response = curl_exec($ch);

                    // Close the cURL session
                    curl_close($ch);
                }
            }
        }

        if ($order->status == "Delivery Done") {
            // Get order details and product size variations associated with the order
            $orderDetails = OrderDetail::where("order_id", $order->id)->get();

            // Update stock for each product based on order details
            foreach ($orderDetails as $orderDetail) {
                $product = Product::findOrFail($orderDetail->product_id);
                $sizeId = $orderDetail->size_id;
                $newStock = $product->stock - $orderDetail->quantity;
                // Ensure product stock doesn't go negative
                $product->stock = max($newStock, 0);
                $product->save();

                // Update stock for product size variations
                $productSizeVariation = ProductSizeVariation::where(
                    "product_id",
                    $product->id
                )
                    ->where("size_id", $sizeId)
                    ->first();
                if ($productSizeVariation) {
                    $newSizeStock =
                        $productSizeVariation->stock - $orderDetail->quantity;
                    // Ensure size stock doesn't go negative
                    $productSizeVariation->stock = max($newSizeStock, 0);
                    $productSizeVariation->save();
                }

                // Update stock for product size variations
            }
        }

        $status = $order->save();
        if ($status) {
            request()
                ->session()
                ->flash("success", "Order successfully update!");
        } else {
            request()
                ->session()
                ->flash("error", "Error while updating Order status");
        }
        return redirect()->back();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        // Get the user ID from the request
        $userId = $request->user()->id;

        // Query the user based on the user ID
        $user = User::find($userId);

        // Retrieve staff users
        $results = User::where("role", "staff")->get();

        // Fetch products with their size variations
        $products = Product::all();

        // Pass data to the view
        return view(
            "backend.sales.create",
            compact("products", "user", "results")
        );
    }

    public function pos(Request $request)
    {
        // Get the user ID from the request
        $userId = $request->user()->id;

        // Query the user based on the user ID
        $user = User::find($userId);

        // Retrieve staff users
        $results = User::where("role", "staff")->get();

        // Fetch products with their size variations
        $products = Product::all();
        return view('pos', compact("products", "user", "results"));
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //return $request->all();

        $this->validate($request, [
            "customerName" => "required",
            "customerPhone" => "required",
            "customerAddress" => "required",
            "sub_total" => "required",
            "shipping_cost" => "required",
            "discount" => "required",
            "total_amount" => "required",
            "fbwaorder" => "",
        ]);

        $general = Setting::where("id", 1)->first();

        $order = new Order();
        $order_data = $request->all();
        $order_data["order_number"] = $general->order_prefix . strtoupper(Str::random(10));
        $order_data["shipping_cost"] = $request->shipping_cost;
        $order_data["sub_total"] = $request->sub_total;
        $order_data["total_amount"] = $request->total_amount;
        $order_data["discount"] = $request->discount;
        $order_data["payment"] = $request->payment;
        $order_data["status"] = "New Order";
        $order_data["payment_method"] = "cod";
        $order_data["payment_status"] = "Unpaid";
        $order_data["first_name"] = $request->customerName;
        $order_data["phone"] = $request->customerPhone;
        $order_data["address"] = $request->customerAddress;
        $order_data["note"] = $request->customerNote;
        $order_data["fbwaorder"] = $request->facebook_order;

        $user = User::where('role', 'staff')->inRandomOrder()->first();
        if ($user) {
            $order_data['assign_user_id'] = $user->id;
        } else {
            $order_data['assign_user_id'] = 1;
        }

        $order->fill($order_data);
        //dd($request->order);
        $status = $order->save();
        if (!Auth::check()) {
            $user = User::firstOrCreate(
                ['phone' => $order->phone],
                [
                    'name' => $order->first_name,
                    'verified' => 1,
                    'current_address' => $request->customerAddress,
                    'password' => Hash::make('12345678')
                ]
            );
        }

        // Assuming $order is already defined somewhere in your code
        $order = Order::findOrFail($order->id); // Fetch the specific Order using findOrFail

        $data = ['user_id' => auth()->user()->id]; // Prepare data to update

        $order->update($data); // Update the Order record with the new user_id
        foreach ($request->order as $key => $row) {
            //dd($row['product_id']);
            $product = Product::where("id", $row["product_id"])->first();
            $cart = new OrderDetail();
            $cart->product_id = $product->id;
            $cart->order_id = $order->id;
            $cart->cost = $product->cost;
            $cart->price = $product->offer_price;
            $cart->quantity = $row["quantity"];
            if (isset($row["size"])) {
                $cart->size = $row["size"];
            }
            if (isset($row["color"])) {
                $cart->color = $row["color"];
            }
            $cart->amount = $product->offer_price * $row["quantity"];
            $cart->save();
        }

        if ($status) {
            request()
                ->session()
                ->flash("success", "Successfully added");
        } else {
            request()
                ->session()
                ->flash("error", "Please try again!!");
        }
        return redirect()->route("order.index");
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $data = Order::findOrfail($id);
        $phone = $data->phone;
        // Make the API request
        $response = Http::get("https://redx.com.bd/api/redx_se/admin/parcel/customer-success-return-rate", [
            'phoneNumber' => '88' . $phone,
        ]);
        // Initialize variables for the API data
        $totalParcels = null;
        $deliveredParcels = null;
        $returnPercentage = null;
        $customerSegment = null;
        if ($response->successful()) {
            $apiData = $response->json();

            // Extract the necessary information
            $totalParcels = $apiData['data']['totalParcels'];
            $deliveredParcels = $apiData['data']['deliveredParcels'];
            $returnPercentage = $apiData['data']['returnPercentage'];
            $customerSegment = $apiData['data']['customerSegment'];
        } else {
            // Handle the error
            return response()->json(['error' => 'Failed to fetch data from API'], 500);
        }
        $created_at = $data->created_at;
        $day = $created_at->day;
        $month = $created_at->month;
        $year = $created_at->year;
        $order_id = $data->id;

        $invoice = $day . $month . $year . $order_id;
        $products = Order::orderBy("created_at", "DESC")->paginate(10);
        $product = OrderDetail::where("order_id", $id)
            ->leftJoin(
                "products",
                "order_details.product_id",
                "=",
                "products.id"
            )
            ->select(
                "order_details.*",
                "products.title as product_title",
                "products.photo as product_photo"
            )
            ->get();
        //dd($product);
        return view(
            "backend.sales.show",
            compact("data", "product", "invoice", "products", "totalParcels", "deliveredParcels", "returnPercentage", "customerSegment")
        );
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id, Request $request)
    {
        // Get the user ID from the request
        $userId = $request->user()->id;

        // Query the user based on the user ID
        $user = User::find($userId);

        // Retrieve staff users
        $results = User::where("role", "staff")->get();

        // Fetch order data
        $order = Order::with('orderDetails.product.sizeVariations.size')->findOrFail($id);

        // Get additional data for the view
        $areas = $this->getRedxAreaList();
        $cities = $this->getPathaoCityList();
        $stores = $this->getPathaoStoreList();
        $couriers = Courier::all();
        $products = Product::with('sizeVariations.size')->get();

        // Pass data to the view
        return view(
            "backend.sales.edit",
            compact(
                "order",
                "products",
                "areas",
                "cities",
                "couriers",
                "stores",
                "results",
                "user"
            )
        );
    }

    public function print($id)
    {
        $data = Order::findOrfail($id);
        $created_at = $data->created_at;
        $day = $created_at->day;
        $month = $created_at->month;
        $year = $created_at->year;
        $order_id = $data->id;

        $invoice = $day . $month . $year . $order_id;

        $product = OrderDetail::where("order_id", $id)
            ->leftJoin(
                "products",
                "order_details.product_id",
                "=",
                "products.id"
            )
            ->select("order_details.*", "products.title as product_title")
            ->get();
        //dd($product);
        return view(
            "backend.report.invoice",
            compact("data", "product", "invoice")
        );
    }
    public function bulkPrints(Request $request)
    {
        // Retrieve selected order IDs from the form submission
        $orderIds = $request->input("bulk_print_order_id", []);

        // Check if any orders are selected
        if (!is_array($orderIds) || empty($orderIds)) {
            return redirect()
                ->back()
                ->with("error", "Please select at least one order.");
        }

        // Prepare an array to hold invoices data
        $invoices = [];

        foreach ($orderIds as $orderId) {
            // Retrieve the order based on the selected order ID
            $order = Order::findOrFail($orderId);

            // Retrieve order details for the current order
            $orderDetails = OrderDetail::where("order_id", $orderId)->get();

            // Initialize an empty array to hold product details
            $products = [];

            foreach ($orderDetails as $detail) {
                // Retrieve product size variation based on order detail
                $product_size_id = ProductSizeVariation::where("product_id", $detail->product_id)
                    ->where("size_id", $detail->size_id)
                    ->first();

                // Fetch product details using join
                $product = OrderDetail::where("order_id", $orderId)
                    ->join("products", "order_details.product_id", "=", "products.id")
                    ->select("order_details.*", "products.title as product_title", "products.photo as product_photo")
                    ->where("order_details.id", $detail->id)
                    ->first();


                // Add product details to the products array
                $products[] = [
                    "detail" => $detail,
                    "product" => $product,
                    "product_size_id" => $product_size_id,
                ];
            }

            // Calculate invoice number
            $invoiceNumber = $order->created_at->format("dmY") . $order->id;

            // Push invoice data to invoices array
            $invoices[] = [
                "order" => $order,
                "products" => $products,
                "invoiceNumber" => $invoiceNumber,
            ];
        }

        // Pass data to the view
        return view("backend.report.bulkindex", compact("invoices"));
    }


    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //dd($request->all());

        $this->validate($request, [
            "customerName" => "required",
            "customerPhone" => "required",
            "customerAddress" => "required|min:10", // Chained the rules together correctly
            "sub_total" => "required",
            "shipping_cost" => "required",
            "discount" => "required",
            "total_amount" => "required",
            "assign_user_id" => "",
            "store_id" => "",
            "courier_id" => " ", // Added a validation rule for courier_id
        ]);

        $general = Setting::where("id", 1)->first();
        // Get the user ID from the request
        $userId = $request->user()->id;

        // Query the user based on the user ID
        $user = User::find($userId);

        $order = Order::findOrFail($id);
        $order_data = $request->all();
        $order_data["order_number"] =
            $general->order_prefix . strtoupper(Str::random(10));
        $order_data["shipping_cost"] = $request->shipping_cost;
        $order_data["sub_total"] = $request->sub_total;
        $order_data["total_amount"] = $request->total_amount;
        $order_data["discount"] = $request->discount;
        $order_data["collectables"] = $request->collectables;
        $order_data["payment"] = $request->payment;
        $order_data["payment_status"] = $request->payment_status;
        $order_data["first_name"] = $request->customerName;
        $order_data["phone"] = $request->customerPhone;
        $order_data["address"] = $request->customerAddress;
        $order_data["note"] = $request->customerNote;
        $order_data["trxid"] = $request->trxid;
        $order_data["bank_trxid"] = $request->bank_trxid;
        $order_data["redx_area_id"] = $request->redx_area_id;
        $order_data["redx_area"] = $request->area_name;
        $order_data["courier_tracking_id"] = $request->courier_tracking_id;
        $order_data["pathao_city"] = $request->pathao_city;
        $order_data["pathao_state"] = $request->pathao_state;
        $order_data["pathao_area_id"] = $request->pathao_area_id;
        $order_data["weight"] = $request->weight;
        $order_data["courier_id"] = $request->courier_id;
        $order_data["store_id"] = $request->store_id;
        $order_data["assign_user_id"] = $request->assign_user_id
            ? $request->assign_user_id
            : $user->id;

        $order->fill($order_data);
        $status = $order->save();

        if (isset($order)) {
            $sub = OrderDetail::where("order_id", $order->id)->get();
            foreach ($sub as $key => $value) {
                $finsub = OrderDetail::findOrFail($value->id);
                $finsub->delete();
            }
        }

        foreach ($request->order as $row) {
            // Ensure product_id and quantity exist
            if (!isset($row["product_id"]) || !isset($row["quantity"])) {
                continue; // Skip if essential data is missing
            }

            $product = Product::findOrFail($row["product_id"]);
            $cart = new OrderDetail();

            $cart->product_id = $product->id;
            $cart->order_id = $order->id;
            $cart->cost = $product->cost;
            $cart->price = $product->offer_price;
            $cart->quantity = $row["quantity"];

            // Set optional fields
            $cart->size = $row["size"] ?? null; // Use null if not set
            $cart->color = $row["color"] ?? null; // Use null if not set

            $cart->amount = $product->offer_price * $row["quantity"];
            $cart->save();
        }


        if ($status) {
            request()
                ->session()
                ->flash("success", "Order Successfully updated");
        } else {
            request()
                ->session()
                ->flash("error", "Please try again!!");
        }
        return redirect()->route("order.index");
    }
    public function custommessage(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        $data = Api::where("id", 1)->first();
        if ($data->sms_status == "on") {
            $settings = Setting::where("id", 1)->first();
            $message = $request->message;
            $orderprefix = $settings->order_prefix;
            $orderid = $settings->order_prefix . $order->id;
            $amount = $order->total_amount;
            $sub_total = $order->sub_total;
            $discount = $order->discount;
            $collectables = $order->payment - $order->total_amount;
            $shippingcost = $order->shipping_cost;
            $first_name = $order->first_name;
            $phone = $order->phone;
            $address = $order->address;

            $message = str_replace("{id}", $orderid, $message);
            $message = str_replace("{amount}", $amount, $message);
            $message = str_replace("{sub_total}", $sub_total, $message);
            $message = str_replace("{discount}", $discount, $message);
            $message = str_replace("{shipping_cost}", $shippingcost, $message);
            $message = str_replace("{name}", $first_name, $message);
            $message = str_replace("{phone}", $phone, $message);
            $message = str_replace("{address}", $address, $message);
            $message = str_replace("{collectables}", $collectables, $message);

            $phonenumber = $order->phone;
            $message = urlencode($message);
            //dd($message);

            $phoneNumber = "88" . $request->phone; // The phone number to send the message to
            //dd($phoneNumber);
            $senderId = $data->sms_sender_id; // Your approved Sender ID
            $apiKey = $data->sms_api_key; // Your API key

            // Build the URL
            $url =
                "http://bulksmsbd.net/api/smsapi?api_key=" .
                $apiKey .
                "&type=text&number=" .
                $phoneNumber .
                "&senderid=" .
                $senderId .
                "&message=" .
                $message;

            // Initialize cURL
            $ch = curl_init();

            // Set cURL options
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

            // Execute the request
            $response = curl_exec($ch);

            // Close the cURL session
            curl_close($ch);
            if ($response === false) {
                request()
                    ->session()
                    ->flash("error", curl_error($ch));
            } else {
                request()
                    ->session()
                    ->flash("success", "SMS Sent");
            }
            return redirect()->back();
        } else {
            request()
                ->session()
                ->flash("error", "SMS API is Off!");
        }
        return redirect()->back();
    }
    public function paymentrequest(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        $data = $request->all();
        $data["order_id"] = $id;
        $status = PaymentRequest::create($data);
        if ($status) {
            $smsapi = Api::where("id", 1)->first();
            if ($smsapi->sms_status == "on" && $request->sentsms == "Yes") {
                $settings = Setting::where("id", 1)->first();
                $message = $request->payment_for;
                $phonenumber = $order->phone;
                $message = urlencode($message);

                $phoneNumber = "88" . $request->phone; // The phone number to send the message to
                //dd($phoneNumber);
                $senderId = $smsapi->sms_sender_id; // Your approved Sender ID
                $apiKey = $smsapi->sms_api_key; // Your API key

                // Build the URL
                $url =
                    "http://bulksmsbd.net/api/smsapi?api_key=" .
                    $apiKey .
                    "&type=text&number=" .
                    $phoneNumber .
                    "&senderid=" .
                    $senderId .
                    "&message=" .
                    $message;

                // Initialize cURL
                $ch = curl_init();

                // Set cURL options
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                // Execute the request
                $response = curl_exec($ch);

                // Close the cURL session
                curl_close($ch);
                if ($response === false) {
                    request()
                        ->session()
                        ->flash("error", curl_error($ch));
                } else {
                    request()
                        ->session()
                        ->flash(
                            "success",
                            "Payment request sent successfully with SMS"
                        );
                }
            } else {
                request()
                    ->session()
                    ->flash(
                        "success",
                        "SMS API is Off! But payment request sent successfully"
                    );
            }
        } else {
            request()
                ->session()
                ->flash("error", "Please try again!!");
        }
        return redirect()->back();
    }
    public function paymentremove(Request $request, $id)
    {
        $order = PaymentRequest::findOrFail($id);

        $status = $order->delete();
        if ($status) {
            request()
                ->session()
                ->flash("success", "Payment request Deleted successfully");
        } else {
            request()
                ->session()
                ->flash("error", "Please try again!!");
        }
        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $order = Order::findOrFail($id);
        if (isset($category)) {
            $sub = OrderDetail::where('order_id', $order->id)->get();
            foreach ($sub as $key => $value) {
                $finsub = OrderDetail::findOrFail($value->id);
                $finsub->delete();
            }
        }

        $status = $order->delete();

        if ($status) {
            request()->session()->flash('success', 'Order Successfully Deleted');
        } else {
            request()->session()->flash('error', 'Error While Deleting Order');
        }
        return redirect()->route('order.index');
    }

    //steafast
    public function sendSteadFast(Request $request)
    {
        $print_order_id = $request->print_order_id;


        if (gettype($print_order_id) != "array") {
            request()
                ->session()
                ->flash("error", "Select Order First");
            return back();
        }

        foreach ($print_order_id as $id) {
            $item = Order::find($id);
            if ($item->courier_tracking_id != null) {
                request()
                    ->session()
                    ->flash("error", "This order already send to Courier");
                return back();
            } elseif ($item->courier_tracking_id == null) {
                $status = $this->createSteadfastParcel($item);
                if (!empty($status["consignment"]["consignment_id"])) {
                    $item->courier_tracking_id =
                        $status["consignment"]["consignment_id"];
                    $item->save();
                } else {
                    request()
                        ->session()
                        ->flash("error", "Something went wrong!!");
                    return back();
                }
            }
        }

        request()
            ->session()
            ->flash("success", "Order Send to Steadfast Successfully!!");
        return back();
    }

    function createSteadfastParcel($item)
    {
        $info = SteadFast::first();
        $settings = DB::table("settings")
            ->where("id", 1)
            ->first();

        $response = Http::withHeaders([
            "Api-Key" => $info->steadfast_api_key,
            "Secret-Key" => $info->steadfast_secret_key,
            "Content-Type" => "application/json",
        ])->post($info->api_url . "create_order", [
            "invoice" => $settings->order_prefix . "" . $item->id,
            "recipient_name" => $item->first_name,
            "recipient_phone" => $item->phone,
            "recipient_address" => $item->address,
            "cod_amount" => (int) $item->total_amount - $item->payment,
            "note" => $item->note,
        ]);

        return $response->json();
    }

    //redx

    public function createRedxParcel(Request $request)
    {
        // Retrieve the selected order IDs from the request
        $printOrderIds = $request->print_order_id_get_redx;

        // Check if any orders are selected
        if (!is_array($printOrderIds) || empty($printOrderIds)) {
            return redirect()
                ->back()
                ->with("error", "Please select at least one order.");
        }

        // Retrieve Redx information
        $redxInfo = Redx::first();

        // Loop through each selected order
        foreach ($printOrderIds as $orderId) {
            // Retrieve the order from the database
            $order = Order::find($orderId);

            // Check if the order exists
            if (!$order) {
                return redirect()
                    ->back()
                    ->with("error", "Order not found.");
            }

            // Check if the order has already been sent to Redx
            if ($order->courier_tracking_id !== null) {
                return redirect()
                    ->back()
                    ->with(
                        "error",
                        "Order " .
                            $orderId .
                            " has already been sent to Courier."
                    );
            }

            // Create Redx parcel
            $status = $this->sendcreateRedxParcels($order, $redxInfo);
            //dd($status);

            // Handle the response
            if (isset($status["tracking_id"])) {
                // Update the order's courier_tracking_id with the obtained tracking ID
                $order->courier_tracking_id = $status["tracking_id"];
                $order->save();
            } elseif (isset($status["message"])) {
                return redirect()
                    ->back()
                    ->with(
                        "error",
                        "Error creating Redx parcel: " . $status["message"]
                    );
            } else {
                return redirect()
                    ->back()
                    ->with(
                        "error",
                        "Something went wrong creating Redx parcel for order " .
                            $orderId .
                            "."
                    );
            }
        }

        // Redirect with success message
        return redirect()
            ->back()
            ->with("success", "Orders sent to Redx successfully.");
    }

    private function sendcreateRedxParcels($order, $redxInfo)
    {
        // Send HTTP request to create Redx parcel
        $response = Http::withHeaders([
            "API-ACCESS-TOKEN" => "Bearer " . $redxInfo->redx_api_access_token,
            "Content-Type" => "application/json",
        ])->post($redxInfo->redx_api_base_url . "parcel", [
            "customer_name" => $order->first_name,
            "customer_phone" => $order->phone,
            "delivery_area" => $order->redx_area,
            "delivery_area_id" => $order->redx_area_id,
            "customer_address" => $order->address,
            "merchant_invoice_id" => $order->order_number,
            "cash_collection_amount" =>
            (int) $order->total_amount - $order->payment,
            "parcel_weight" => "500",
            "instruction" => "",
            "value" => $order->total_amount,
            "pickup_store_id" => 0,
            "parcel_details_json" => [],
        ]);

        // Return the response as JSON
        return $response->json();
    }

    function getRedxAreaList()
    {
        // Fetch the first record from the Redx model
        $redxInfo = Redx::first();

        // Send HTTP GET request to Redx API for fetching areas
        $response = Http::withHeaders([
            "API-ACCESS-TOKEN" => "Bearer " . $redxInfo->redx_api_access_token,
        ])->get($redxInfo->redx_api_base_url . "areas");

        // Parse the response JSON and extract the 'areas' array
        $areas = $response->json()["areas"];

        // Return the list of areas
        return $areas;
    }

    //Pathao
    public function OrderSendToPathao(Request $request)
    {
        // Retrieve the IDs of selected orders for Pathao
        $printOrderIds = $request->print_order_id_get_pathao;

        // Check if any orders are selected
        if (!is_array($printOrderIds) || empty($printOrderIds)) {
            return redirect()
                ->back()
                ->with("error", "Please select at least one order.");
        }

        // Retrieve Pathao information
        $pathaoInfo = Pathao::first();
        $settings = DB::table("settings")
            ->where("id", 1)
            ->first();

        // Loop through each selected order and send it to Pathao
        foreach ($printOrderIds as $orderId) {
            // Find the order
            $order = Order::find($orderId);

            // Check if the order exists
            if (!$order) {
                return redirect()
                    ->back()
                    ->with("error", "Order not found.");
            }

            // Check if the order has already been sent to Pathao
            if ($order->courier_tracking_id !== null) {
                return redirect()
                    ->back()
                    ->with(
                        "error",
                        "Order " .
                            $orderId .
                            " has already been sent to Courier."
                    );
            }

            // Create Pathao parcel
            $status = $this->createPathaoParcel($order, $pathaoInfo, $settings);

            // Handle the response
            if (!empty($status["data"]["consignment_id"])) {
                // Update the order's courier_tracking_id with the obtained tracking ID
                $order->courier_tracking_id = $status["data"]["consignment_id"];
                $order->save();
            } elseif (isset($status["message"])) {
                return redirect()
                    ->back()
                    ->with(
                        "error",
                        "Error creating Pathao parcel: " . $status["message"]
                    );
            } else {
                return redirect()
                    ->back()
                    ->with(
                        "error",
                        "Something went wrong creating Pathao parcel for order " .
                            $orderId .
                            "."
                    );
            }
        }

        // Redirect with success message
        return redirect()
            ->back()
            ->with("success", "Orders sent to Pathao successfully.");
    }

    private function createPathaoParcel($order, $pathaoInfo, $settings)
    {
        // Send HTTP POST request to create Pathao parcel
        $response = Http::withHeaders([
            "Authorization" => "Bearer " . $pathaoInfo->pathao_api_access_token,
            "Content-Type" => "application/json",
            "Accept" => "application/json",
        ])->post($pathaoInfo->pathao_api_base_url . "/aladdin/api/v1/orders", [
            "store_id" => $order->store_id,
            "merchant_order_id" => $order->order_number,
            "sender_name" => "Admin",
            "sender_phone" => $settings->phone,
            "recipient_name" => $order->first_name,
            "recipient_phone" => $order->phone,
            "recipient_address" => $order->address,
            "recipient_city" => $order->pathao_city,
            "recipient_zone" => $order->pathao_state,
            "recipient_area" => $order->pathao_area_id,
            "delivery_type" => 48,
            "item_type" => 2,
            "special_instruction" => "",
            "item_quantity" => 1,
            "item_weight" => $order->weight,
            "amount_to_collect" => (int) $order->total_amount,
            "item_description" => $order->note,
        ]);

        // Return the response as JSON
        return $response->json();
    }

    public function statusupdate(Request $request)
    {
        // Verify webhook signature if needed
        // $receivedSignature = $request->header('X-PATHAO-Signature');
        // $expectedSignature = '<123456789abcdeff>';
        // if ($receivedSignature !== $expectedSignature) {
        //     return response()->json(['error' => 'Unauthorized'], 401);
        // }

        // Extract data from the webhook payload

        $payload = $request->all();
        //dd($payload);

        // Log the received payload for debugging
        Log::info("Received webhook payload:", $payload);

        // Process the payload
        if (isset($payload["order_status"])) {
            // Update order status in your application database
            $orderStatus = $payload["order_status"];
            $orderId = $payload["merchant_order_id"];

            // Example: Update order status in database
            // Order::where('id', $orderId)->update(['status' => $orderStatus]);

            // Log successful processing of the payload
            Log::info(
                "Order status updated successfully for order ID: " .
                    $orderId .
                    ". New status: " .
                    $orderStatus
            );

            // Respond with a success message
            return view("statusupdate")->with(
                "success",
                "Order status updated successfully."
            );
        } else {
            // Log an error if the payload is missing required data
            Log::error("Invalid webhook payload: Missing order_status");

            // Respond with an error message
            return view("statusupdate")->with(
                "error",
                "Invalid webhook payload: Missing order_status"
            );
        }
    }

    public function getPathaoStoreList()
    {
        // Retrieve Pathao information
        $pathaoInfo = Pathao::first();

        // Initialize an empty array to store the list of stores
        $stores = [];

        try {
            // Send HTTP GET request to Pathao API for fetching stores
            $response = Http::withHeaders([
                "Authorization" =>
                "Bearer " . $pathaoInfo->pathao_api_access_token,
            ])->get(
                $pathaoInfo->pathao_api_base_url . "/aladdin/api/v1/stores"
            );

            // Check if the response is successful
            if ($response->successful()) {
                // Parse the response JSON and extract the store data
                $stores = $response->json()["data"]["data"] ?? [];
            } else {
                // Log the error if the request was not successful
                \Log::error(
                    "Failed to fetch Pathao stores: " . $response->status()
                );
            }
        } catch (\Exception $e) {
            // Log any exceptions that occur during the request
            \Log::error("Error fetching Pathao stores: " . $e->getMessage());
        }

        // Return the list of stores
        return $stores;
    }

    //Pathao Courier Service

    function getPathaoCityList()
    {
        // Retrieve Pathao information
        $pathaoInfo = Pathao::first();
        $response = Http::withHeaders([
            "Authorization" => "Bearer " . $pathaoInfo->pathao_api_access_token,
        ])->get(
            $pathaoInfo->pathao_api_base_url .
                "/aladdin/api/v1/countries/1/city-list"
        );

        $cities = $response->json()["data"]["data"];
        return $cities;
    }

    function getPathaoZoneListByCity($city)
    {
        // Retrieve Pathao information
        $pathaoInfo = Pathao::first();
        $response = Http::withHeaders([
            "Authorization" => "Bearer " . $pathaoInfo->pathao_api_access_token,
        ])->get(
            $pathaoInfo->pathao_api_base_url .
                "/aladdin/api/v1/cities/" .
                $city .
                "/zone-list"
        );

        $zones = $response->json()["data"]["data"];

        return response()->json(["success" => true, "zones" => $zones]);
    }

    function getPathaoAreaListByZone($zone)
    {
        // Retrieve Pathao information
        $pathaoInfo = Pathao::first();
        $response = Http::withHeaders([
            "Authorization" => "Bearer " . $pathaoInfo->pathao_api_access_token,
        ])->get(
            $pathaoInfo->pathao_api_base_url .
                "/aladdin/api/v1/zones/" .
                $zone .
                "/area-list"
        );

        $areas = $response->json()["data"]["data"];

        return response()->json(["success" => true, "areas" => $areas]);
    }

    public function viewAccessToken()
    {
        return view("backend.api.generate-pathao-access-token");
    }

    public function generatePathaoAccessToken(Request $request)
    {
        $pathaoInfo = Pathao::first();
        $response = Http::withHeaders([
            "content-type" => "application/json",
            "accept" => "application/json",
        ])->post(
            $pathaoInfo->pathao_api_base_url . "/aladdin/api/v1/issue-token",
            [
                "client_id" => $request->client_id,
                "client_secret" => $request->client_secret,
                "username" => $request->client_email,
                "password" => $request->client_password,
                "grant_type" => "password",
            ]
        );

        dd($response->json());
    }

}
